<?php
    require __DIR__ . "/User.php";

    //TODO: Search for users by job type, supervisor, and phone number

    /**
     * For ease of User management
     *
     * @author JoshuaA
     *
     */
    class UserManager {
        /**
         *
         * @param mysqli $db
         * @return void
         */
        function __construct($db, $edb) {
            $this->db = $db;
            $this->edb = $edb;
        }

        /**
         * Adds a new user to the database
         *
         * @param array<string, string|int> $opts [
         *     "email" => string,
         *     "first_name" => string,
         *     "last_name" => string,
         *     "password" => string,
         *     "job_id" => int,
         *     "user_type" => int,
         *     "supervisor_id" => int,
         *     "phone_number" => (optional)string,
         *     "birthday" => (optional)string (YYYY-MM-DD)
         * ]
         * @throws Exception "Invalid parameters"
         * @throws Exception "User already exists"
         * @throws Exception "User not found" (shouldn't happen but you never know)
         * @return User
         */
        function createUser($opts) { //Likely won't be using //TODO: Update
            if (!isset($opts["email"]) || //?Do this a different way? Seems inefficient
            !isset($opts["first_name"]) ||
            !isset($opts["last_name"]) ||
            !isset($opts["password"]) ||
            !isset($opts["job_id"]) ||
            !isset($opts["user_type"]) ||
            !isset($opts["supervisor_id"])) throw new Exception("Invalid parameters");

            if ($this->userExists($opts["email"])) throw new Exception("User already exists");

            $sql = "insert into phone_users (email, first_name, last_name, password, job_id, user_type, supervisor_id";
            $values = "(?, ?, ?, ?, ?, ?, ?";
            $enc_pass = password_hash($opts["password"], PASSWORD_DEFAULT); //*using password_hash is most secure. neary impossible to decrypt. nobody can view the password, only attempt to brute force it

            $vars = array(
                "ssssiii",
                $opts["email"],
                $opts["first_name"],
                $opts["last_name"],
                $enc_pass,
                $opts["job_id"],
                $opts["user_type"],
                $opts["supervisor_id"]
            );

            if (isset($opts["phone_number"])) {
                array_push($vars, $opts["phone_number"]);
                $sql .= ", phone_number";
                $values .= ", ?";
                $vars[0] .= "s";
            }

            if (isset($opts["birthday"])) {
                array_push($vars, $opts["birthday"]);
                $sql .= ", birthday";
                $values .= ", ?";
                $vars[0] .= "s";
            }

            $sql .= ") values " . $values . ")";

            runIQuery($this->db, $sql, $vars);

            return $this->findUser($opts["email"]);
        }

        /**
         * Gets all users in the database as an array of User objects or sanitized data
         *
         * @param boolean $sanitized (optional) If true will return array of users with minimal data
         * @see User::getSanitizedData()
         * @return array<int, User>|array<int, array<string, string|int|null>>
         */
        function getAllUsers($sanitized = false) {
            $all = $this->findUsers("");

            if ($sanitized) {
                $users = array();
                foreach ($all as $user) array_push($users, $user->getSanitizedData());
                return $users;
            }

            return $all;
        }

        /**
         * Gets an array of a given user's workers as an array of User objects or sanitized data
         *
         * @param User|string|int $user User object or email or user_id
         * @param boolean $sanitized(optional) If true will return array of users with minimal data
         *
         * @throws Exception "User not found"
         * @see User::getSanitizedData()
         * @return array<int, User>|array<int, array<string, string|int|null>>
         */
        function getWorkersOf($user, $sanitized = false) {
            if (!($user instanceof User)) $user = $this->findUser($user);

            $sql = "select email from phone_users where supervisor_id=?";
            list($rs) = runIQuery($this->db, $sql, array("i", $user->id));

            $workers = array();
            if (count($rs) == 0) return $workers;

            foreach ($rs as $user) {
                $user["usermanager"] = $this;
                $user = $this->findUser($user["email"]);
                if ($sanitized) $user = $user->getSanitizedData();
                array_push($workers, $user);
            }

            return $workers;
        }

        /**
         * Gets a given user's supervisor as a User object or sanitized data
         *
         * @param User|string|int $user User object or email or user_id
         * @param boolean $sanitized (optional) If true will return array of users with minimal data
         * @see User::getSanitizedData()
         * @return User|array<string, string|int|null>
         */
        function getSupervisorOf($user, $sanitized = false) {
            if (!($user instanceof User)) $user = $this->findUser($user);

            return $this->findUser($user->supervisor_email);
        }

        /**
         * Attempts to find a user by email or user_id
         *
         * @param string|int $search email or user_id
         * @throws Exception "User not found"
         * @return User
         */
        function findUser($search) {
            $sql = "select * from phone_users where user_id=? or email=? limit 1";

            list($rs) = runIQuery($this->db, $sql, array("is", $search, $search));

            if (count($rs) == 0) {
                throw new Exception("User not found");
                return;
            }

            $sql = "select d.department_name, d.lead_email as `lead`, d.department_id, e.first_name, e.last_name from departments d, employees e where e.employee_email=? and e.department_id=d.department_id";
            list($ers) = runIQuery($this->edb, $sql, ["s", $rs[0]["email"]]);
            if (!count($ers)) {
                throw new Exception("User not found");
                return;
            }

            $auser = $rs[0];
            $euser = $ers[0];
            $user = [
                "user_id" => $auser["user_id"],
                "email" => $auser["email"],
                "password" => $auser["password"],
                "last_seen" => $auser["last_seen"],
                "first_name" => $euser["first_name"],
                "last_name" => $euser["last_name"],
                "is_online" => $auser["is_online"],
                "department_id" => $euser["department_id"],
                "department_name" => $euser["department_name"],
                "supervisor_email" => $auser["supervisor_email"],
                "supervisor_id" => $auser["supervisor_id"],
                "is_supervisor" => $auser["is_supervisor"],
                "usermanager" => $this
            ];

            return new User($this->db, $user);
        }

        /**
         * Attempts to find a token's user as a User object
         *
         * @param Token|string $token
         * @throws Exception "Token not found"
         * @return User
         */
        function findUserByToken($token) { //!Unused
            if (!($token instanceof Token)) $token = Token::getFromTokenString($this->usermanager, $token);
            return $token->user;
        }

        /**
         * Attempts to find users by first name, last name, or full name
         *
         * @param string $name first name, last name, or full name
         * @param boolean $sanitized (optional) If true will return array of users with minimal data
         * @see User::getSanitizedData()
         * @return array<int, User>|array<int, array<string, string|int|null>>
         */
        function findUsers($email, $sanitized = false) {

            $sql = "select * from phone_users where email like ?";
            list($rs) = runIQuery($this->db, $sql, ["s", "%$email%"]);

            $users = array();
            $rs = stripResults($rs);
            foreach ($rs as $user) {
                $user["usermanager"] = $this;
                $user = $this->findUser($user["email"]);
                if ($sanitized) $user = $user->getSanitizedData();
                array_push($users, $user);
            }

            return $users;
        }

        /**
         * Checks if a user exists in the database
         *
         * @param string|int $search email or user_id
         * @return boolean
         */
        function userExists($search) {
            try {
                $this->findUser($this->db, $search);
                $exists = true;
            } catch(Exception $err) {
                $exists = false;
            }

            return $exists;
        }
    }
?>