<?php
    require __DIR__ . "/Token.php";
    require __DIR__ . "/MessageHandler.php";
    require __DIR__ . "/runIQuery.php";

    /**
     * User class holds all data and useful functions
     *
     * @author JoshuaA
     *
     */
    class User {
        /**
         * Database containing the User
         * @var mysqli
         */
        var $db;
        /**
         * The users user_id
         * @var int
         */
        var $id;
        /**
         * The users email
         * @var string
         */
        var $email;
        /**
         * The users password (hashed and encrypted)
         * @var string
         */
        var $password;
        /**
         * When the user last sent a request to the server (YYYY-MM-DD HH:MM:SS)
         * @var string
         */
        var $last_seen;
        /**
         * Users first name
         * @var string
         */
        var $first_name;
        /**
         * Users last name
         * @var string
         */
        var $last_name;
        /**
         * Users full name, set automatically using $first_name and $last_name
         * @var string
         */
        var $full_name;
        /**
         * If the user is currently active/online
         * @var boolean
         */
        var $is_online;
        /**
         * Users job id
         * @var int
         */
        var $department_id;
        /**
         * Users job name
         * @var string
         */
        var $department_name;
        /**
         * Users supervisors id
         * @var int
         */
        var $supervisor_email;
        /**
         * All of the users valid tokens
         * @see Token
         * @var array<int, Token>
         */
        var $tokens;
        /**
         * Users message handler for sending, getting, and receiving messages
         * @see MessageHandler
         * @var MessageHandler
         */
        var $messages;
        /**
         * Users UserManager
         * @see UserManager
         * Mostly for easy access to $db so it only needs to be defined once in the UserManager
         * @var UserManager
         */
        var $usermanager;

        /**
         *
         * @param mysqli $db
         * @param array<string, string|int> $opts [
         *     "user_id" => int,
         *     "email" => string,
         *     "password" => string,
         *     "last_seen" => string (YYYY-MM-DD HH:MM:SS),
         *     "first_name" => string,
         *     "last_name" => string,
         *     "phone_number" => string/null,
         *     "birthday" => string/null (YYYY-MM-DD),
         *     "active" => bool,
         *     "user_type" => string,
         *     "is_admin" => bool,
         *     "job_id" => int,
         *     "job_name" => string,
         *     "supervisor_id" => int,
         *     "usermanager" => UserManager
         * ]
         * @return void
         */
        function __construct($db, $opts) {

            $this->db = $db;
            $this->id = $opts["user_id"]; //bigint
            $this->email = $opts["email"]; //string
            $this->password = $opts["password"]; //string
            $this->last_seen = $opts["last_seen"]; //string, datetime
            $this->first_name = $opts["first_name"]; //string
            $this->last_name = $opts["last_name"]; //string
            $this->full_name = $this->first_name . " " . $this->last_name; //string
            $this->is_online = $opts["is_online"]; //bool (int, 0 or 1)
            $this->department_id = $opts["department_id"]; //bigint
            $this->department_name = $opts["department_name"]; //string
            $this->supervisor_email = $opts["supervisor_email"]; //bigint
            $this->supervisor_id = $opts["supervisor_id"];
            $this->is_supervisor = $opts["is_supervisor"];

            //!Depricated
            $this->birthday = "2022-05-05";
            $this->phone_number = "1111111111";
            $this->user_type = 1;

            $this->tokens = Token::getValidTokens($this);

            $this->messages = new MessageHandler($this);
            $this->usermanager = $opts["usermanager"];
        }

        /**
         * Removes all existing tokens for the current user
         *
         * @return User $this
         */
        function clearTokens() {
            $sql = "delete from phone_user_tokens where user_id=?";
            runIQuery($this->db, $sql, array("i", $this->id));

            array_splice($this->tokens, 0);

            return $this;
        }

        /**
         * Sets a new password for the user and clears all tokens
         *
         * @param string $pass The new password (unencrypted)
         * @return User $this
         */
        function setPassword($pass) { //!will invalidate all tokens for security
            $sql = "update phone_users set password=? where user_id=?";
            $enc_pass = password_hash($pass, PASSWORD_DEFAULT);

            runIQuery($this->db, $sql, array("si", $enc_pass, $this->id));
            $this->password = $enc_pass;

            $this->clearTokens();

            return $this;
        }

        /**
         * Compares the given password to encrypted password
         *
         * Will rehash the password in the database if PHP's PASSWORD_DEFAULT options have been improved which helps keeps things secure.
         *
         * @param string $pass The password to test
         * @return boolean true if password matches
         */
        function verifyPassword($pass) {
            $matches = password_verify($pass, $this->password);

            if ($matches) {
                if (password_needs_rehash($this->password, PASSWORD_DEFAULT)) { //?rehash the password if PASSWORD_DEFAULT has changed/improved, improves security over time
                    $sql = "update phone_users set password=? where user_id=?";
                    runIQuery($this->db, $sql, array("si", password_hash($pass, PASSWORD_DEFAULT), $this->id));
                }
            }

            return $matches;
        }

        /**
         * Returns minimal data to send to the client
         *
         * @return array<string, string|int|null> [
         *     "user_id" => int,
         *     "email" => string,
         *     "first_name" => string,
         *     "last_name" => string,
         *     "birthday" => string,
         *     "job_title" => string,
         *     "job_id" => int,
         *     "user_type" => string,
         *     "active" => bool,
         *     "supervisor_id" => int
         * ]
         */
        function getSanitizedData() {
            $data = array(
                "user_id" => $this->id,
                "email" => $this->email,
                "first_name" => $this->first_name,
                "last_name" => $this->last_name,
                "birthday" => $this->birthday,
                "job_title" => $this->department_name,
                "department_name" => $this->department_name,
                "job_id" => $this->department_id,
                "department_id" => $this->department_id,
                "user_type" => $this->user_type,
                "is_online" => $this->is_online,
                "supervisor_email" => $this->supervisor_email,
                "supervisor_id" => $this->supervisor_id,
                "is_supervisor" => $this->is_supervisor
            );

            return $data;
        }

        /**
         * Gets the instances supervisor User object or sanitized data
         *
         * @param boolean $sanitized (optional) If true will return array of users with minimal data
         * @see User::getSanitizedData()
         * @return User OR sanitized user data
         */
        function getSupervisor($sanitized = false) {
            return $this->usermanager->getSupervisorOf($this, $sanitized);
        }

        /**
         * Gets workers as an array of User objects or sanitized data
         *
         * @param boolean $sanitized (optional) If true will return array of users with minimal data
         * @see User::getSanitizedData()
         * @return array<int, User>|array<int, array<string, string|int|null>>
         */
        function getWorkers($sanitized = false) {
            return $this->usermanager->getWorkersOf($this, $sanitized);
        }
    }
?>