<?php
    require __DIR__ . "/UserManager.php";

    /**
     * For ease of PTO Request handling
     * @author JoshuaA
     *
     */
    class PTORequest {
        /**
         * The user that the request belongs to
         * @var User
         */
        var $user;
        /**
         * The ID of the request
         * @var int
         */
        var $id;
        /**
         * The start date of the request
         * @var string
         */
        var $date_start;
        /**
         * The end date of the request
         * @var string
         */
        var $date_end;
        /**
         * The optional notes of the request
         * @var string
         */
        var $notes;
        /**
         * The date that the request was created
         * @var string
         */
        var $date_submitted;
        /**
         * The date that the request was reviewed
         * @var string
         */
        var $date_reviewed;
        /**
         * Who reviewed the given request
         * @var int
         */
        var $reviewed_by;
        /**
         * The current status of the request as its ID
         * @var int
         */
        var $status;
        /**
         * The current status of the request as its name
         * @var string
         */
        var $status_name;

        /**
         *
         * @see User
         * @param User $user
         * @param array<string, string|int|null> $opts [
         *     "date_start" => string "YYYY-MM-DD",
         *     "date_end" => string  "YYYY-MM-DD",
         *     "notes" => string|null,
         *     "date_reviewed" => string|null "YYYY-MM-DD HH:MM:SS",
         *     "reviewed_by" => int user_id,
         *     "status" => int
         * ]
         * @return void
         */
        function __construct($user, $opts) {
            if (!isset($opts["notes"])) $opts["notes"] = "";
            if (!isset($opts["date_reviewed"])) $opts["date_reviewed"] = null;
            if (!isset($opts["reviewed_by"])) $opts["reviewed_by"] = null;
            if (!isset($opts["status"])) {
                $opts["status"] = 1;
                $opts["status_name"] = "pending";
            }
            if (!isset($opts["status_name"])) {
                $sql = "select * from phone_pto_status_types where type_id=?";
                list($rs) = runIQuery($user->db, $sql, array("i", $opts["status"]));

                $opts["status_name"] = $rs[0]["type_name"];
            }

            $this->user = $user;
            $this->id = $opts["request_id"];
            $this->date_start = $opts["date_start"];
            $this->date_end = $opts["date_end"];
            $this->notes = $opts["notes"];
            $this->date_submitted = $opts["date_submitted"];
            $this->date_reviewed = $opts["date_reviewed"];
            $this->reviewed_by = $opts["reviewed_by"];
            $this->paid = $opts["paid"];
            $this->status = $opts["status"];
            $this->status_name = $opts["status_name"];
        }

        /**
         * Applies any changes made to the database, use after all setXXXX functions
         *
         * @return PTORequest $this
         */
        function apply() {
            $sql = "update phone_pto_requests set date_start=?, date_end=?, notes=?, date_submitted=?, date_reviewed=?, reviewed_by=?, status=?, paid=? where request_id=" . $this->id;
            list($rs,$err) = runIQuery($this->user->db, $sql, array(
                "sssssiii",
                $this->date_start,
                $this->date_end,
                $this->notes,
                $this->date_submitted,
                $this->date_reviewed,
                $this->reviewed_by,
                $this->status,
                $this->paid
            ));

            return $this;
        }

        /**
         * Sets the notes of the request
         *
         * @param string $notes
         * @return PTORequest $this
         */
        function setNotes($notes) {
            $this->notes = $notes;
            return $this;
        }

        function setPaid($paid) {
            $this->paid = !!$paid;
            return $this;
        }

        /**
         * Sets the status of the request
         *
         * @param string|int $status
         * @return PTORequest $this
         */
        function setStatus($status) {
            $sql = "select * from phone_pto_status_types where type_id=? or type_name=?";
            list($rs,$err) = runIQuery($this->user->db, $sql, array("is", $status, $status));
            $this->status = $rs[0]["type_id"];
            $this->status_name = $rs[0]["type_name"];

            return $this;
        }

        /**
         * Sets the start date of the request
         *
         * @param string $date "YYYY-MM-DD"
         * @return PTORequest $this
         */
        function setStart($date) {
            $this->date_start = $date;
            return $this;
        }

        /**
         * Sets the end date of the request
         *
         * @param string $date "YYYY-MM-DD"
         * @return PTORequest $this
         */
        function setEnd($date) {
            $this->date_end = $date;
            return $this;
        }

        /**
         * Sets the date_reviewed and who reviewed it
         *
         * @param int $user_id
         * @return PTORequest $this
         */
        function setReviewedBy($user_id) {
            global $dtformat;
            $this->date_reviewed = date($dtformat);
            $this->reviewed_by = $user_id;
            return $this;
        }

        /**
         * Returns minimal data to send to the client
         *
         * @return array<string, string|int|null>
         */
        function getSanitizedData() {
            return array(
                "user_id" => $this->user->id,
                "request_id" => $this->id,
                "date_start" => $this->date_start,
                "date_end" => $this->date_end,
                "date_submitted" => $this->date_submitted,
                "date_reviewed" => $this->date_reviewed,
                "reviewed_by" => $this->reviewed_by,
                "notes" => $this->notes,
                "status" => $this->status_name,
                "paid" => $this->paid
            );
        }

        /**
         *
         * @param User $user
         * @param array<string, string|int> $opts
         * @throws Exception
         * @return PTORequest
         */
        public static function createRequest($user, $opts) {
            global $dtformat;
            if (!isset($opts["date_start"]) || !isset($opts["date_end"])) throw new Exception("Invalid parameters");
            if (!isset($opts["notes"])) $opts["notes"] = "";
            if (!isset($opts["date_submitted"])) $opts["date_submitted"] = date($dtformat);
            if (!isset($opts["paid"])) $opts["paid"] = 0;
            if (!isset($opts["status"])) $opts["status"] = 1;
            $sql = "select * from phone_pto_status_types where type_id=?";

            list ($rs) = runIQuery($user->db, $sql, array("i", $opts["status"]));
            $opts["status_name"] = $rs[0]["type_name"];

            $sql = "insert into phone_pto_requests (user_id, date_start, date_end, date_submitted, notes, status, paid) values (?, ?, ?, ?, ?, ?, ?)";

            list($rs,$err) = runIQuery($user->db, $sql, array(
                "issssis",
                $user->id,
                $opts["date_start"],
                $opts["date_end"],
                $opts["date_submitted"],
                $opts["notes"],
                $opts["status"],
                $opts["paid"]
            ));
            $opts["request_id"] = $rs[0]["id"];
            print_r($err);

            return new PTORequest($user, $opts);
        }

        /**
         * Finds a specific request for specific user
         *
         * @param User $user
         * @param int $id request_id
         * @param UserManager $usermanager
         * @return PTORequest
         */
        public static function findRequest($user, $id, $usermanager = false) {
            if (!($user instanceof User)) {
                if (!($usermanager instanceof UserManager)) throw new Exception("No UserManager provided");
                $user = $usermanager->findUser($user);
            }

            $sql = "select * from phone_pto_requests inner join phone_pto_status_types on phone_pto_status_types.type_id=phone_pto_requests.status where request_id=?";
            list($rs) = runIQuery($user->db, $sql, array("i", $id));

            $ruser = $user->usermanager->findUser($rs[0]["user_id"]);

            return new PTORequest($ruser, $rs[0]);
        }

        /**
         * Gets all requests from a given user
         *
         * @param User|string|int $user User|email|user_id
         * @param boolean $usermanager
         * @throws Exception "No UserManager provided"
         * @throws Exception "User not found"
         * @return array<int, PTORequest>
         */
        public static function getRequests($user, $usermanager = false) {
            if (!($user instanceof User)) {
                if (!($usermanager instanceof UserManager)) throw new Exception("No UserManager provided");
                $user = $usermanager->findUser($user);
            }

            $sql = "select * from phone_pto_requests inner join phone_pto_status_types on phone_pto_status_types.type_id=phone_pto_requests.status where user_id=?";
            list($rs) = runIQuery($user->db, $sql, array("i", $user->id));

            $requests = array();
            foreach ($rs as $request) {
                $request["status_name"] = $request["type_name"];
                array_push($requests, new PTORequest($user, $request));
            }

            return $requests;
        }
    }
?>